package com.mth.requestsecret.service;

import com.alibaba.fastjson.JSONObject;
import com.mth.requestsecret.scheduler.RequestSecretSchedulerTask;
import com.mth.requestsecret.util.DSLUtils;
import com.mth.requestsecret.util.MD5Utils;
import com.mth.requestsecret.util.RedisUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.*;
import org.springframework.stereotype.Component;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestTemplate;

import java.util.Date;

import static com.mth.requestsecret.constant.Constants.RESPONSE_CODE_14;
import static com.mth.requestsecret.constant.Constants.SJJ_REQUEST_SECRET_PREFIX;

/**
 * @author fyl
 * @version v1.0
 * @desc
 * @date 2020-07-16 15:00
 * @jdk 1.8
 */
@Component
@Slf4j
public class RestTemlateService {

    @Autowired
    private RestTemplate restTemplate;

    @Autowired
    private RedisUtils redisUtil;

    @Autowired
    private RequestSecretSchedulerTask schedulerTask;

    private ThreadLocal<Integer> threadLocal = ThreadLocal.withInitial(() -> 0);

    @Value("${DATA_API_ADDRESS}")
    private String pathUrl;

    @Value("${app.key}")
    private String appKey;

    public ResponseEntity<String> commonSendRequest(MultiValueMap<String, Object> paramMap, final String apiMethod) {
        // 请求url拼接api签名和公共参数
        StringBuilder url = new StringBuilder()
                .append(pathUrl)
                .append("/interface/public/service/risen-api/")
                .append(apiMethod);

        // 请求时间
        String requestTime = DSLUtils.dateToLong(new Date()) + "";
        // redis中获取秘钥
        String requestSecret = redisUtil.get(SJJ_REQUEST_SECRET_PREFIX);
        log.info("redis中请求秘钥：{}", requestSecret);
        // 签名字符串
        String signStr = appKey + requestSecret + requestTime;
        // 组装请求参数
        paramMap.set("appKey", appKey);
        paramMap.set("sign", MD5Utils.encoderByMd5(signStr));
        paramMap.set("requestTime", requestTime);

        // 日志记录
        log.info("纪委api：{}", apiMethod);
        log.info("api url：{}", url);
        log.info("api params：{}", paramMap);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_FORM_URLENCODED);

        // 发送请求
        HttpEntity<MultiValueMap<String, Object>> request = new HttpEntity<>(paramMap, headers);
        ResponseEntity<String> responseEntity = restTemplate.exchange(url.toString(), HttpMethod.POST, request, String.class);
        log.info("api response：{}", responseEntity);

        try {
            // 接口返回码
            String code = JSONObject.parseObject(responseEntity.getBody()).getString("code");
            Integer count = threadLocal.get();
            String threadName = Thread.currentThread().getName();

            // 如返回签名错误，则重新获取秘钥并发送请求，限制重新请求次数两次
            if (RESPONSE_CODE_14.equals(code) && count < 2) {
                threadLocal.set(++count);
                log.info("线程名称：{}，重新请求次数：{}", threadName, count);
                // 获取秘钥
                schedulerTask.getRequestSecret();
                // 重新调用请求
                return commonSendRequest(paramMap, apiMethod);
            }
        } finally {
            threadLocal.remove();
        }
        return responseEntity;
    }
}