package com.mth.requestsecret.controller;

import com.mth.requestsecret.service.RestTemplateService;
import com.mth.requestsecret.util.AESUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.json.JSONObject;
import org.json.XML;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RestController;

import java.io.ByteArrayInputStream;
import java.nio.charset.StandardCharsets;
import java.text.MessageFormat;
import java.util.HashMap;
import java.util.Map;

/**
 * 审管办接口
 *
 * @author MaTianHao
 * @date 2020/8/27
 */
@RestController
@Slf4j
public class SgbController {

    @Autowired
    private RestTemplateService restTemplateService;

    private static final String TOKEN = "cd5202f4-df21-40d6-8205-db4e326c6521";

    /**
     * 2.1 索引接口
     *
     * @param startTime
     * @param endTime
     * @return
     * @throws Exception
     */
    @PostMapping("/readTenderNoData")
    public ResponseEntity readTenderNoData(String startTime, String endTime) throws Exception {
        String apiMethod = "ReadTenderNoData";

        // 1. 设置参数
        StringBuilder soapRequestData = new StringBuilder();
        appendStartPart(soapRequestData);
        soapRequestData.append("<").append(apiMethod).append(" xmlns=\"http://tempuri.org/\">");
        soapRequestData.append("<userToken>{0}</userToken>");
        soapRequestData.append("<startTime>{1}</startTime>");
        soapRequestData.append("<endTime>{2}</endTime>");
        soapRequestData.append("</").append(apiMethod).append(">");
        appendEndPart(soapRequestData);
        log.info("接收参数 - startTime：{}；endTime：{}", startTime, endTime);

        // 2. 参数加密，替换占位符
        String paramData = MessageFormat.format(soapRequestData.toString(),
                TOKEN, AESUtils.desEncrypt(startTime), AESUtils.desEncrypt(endTime));

        // 3. 发送请求
        String resultStr = restTemplateService.sgbSendRequest(paramData, apiMethod);

        return getParseResponseXmlAndConvertToJson(apiMethod, resultStr);
    }

    /**
     * 2.2 项目基本信息表
     *
     * @param tenderNo
     * @return
     * @throws Exception
     */
    @PostMapping("/readTenderData")
    public ResponseEntity readTenderData(String tenderNo) throws Exception {
        String apiMethod = "ReadTenderData";

        return getResponseEntityByTenderNo(tenderNo, apiMethod);
    }

    /**
     * 2.3 开评标安排表
     *
     * @param tenderNo
     * @return
     * @throws Exception
     */
    @PostMapping("/readProjectArrangementData")
    public ResponseEntity readProjectArrangementData(String tenderNo) throws Exception {
        String apiMethod = "ReadProjectArrangementData";

        return getResponseEntityByTenderNo(tenderNo, apiMethod);
    }

    /**
     * 2.4 开标记录信息表
     *
     * @param tenderNo
     * @return
     * @throws Exception
     */
    @PostMapping("/readOpenBidRecordData")
    public ResponseEntity readOpenBidRecordData(String tenderNo) throws Exception {
        String apiMethod = "ReadOpenBidRecordData";

        return getResponseEntityByTenderNo(tenderNo, apiMethod);
    }

    /**
     * 2.5 评标委员会基本信息表
     *
     * @param tenderNo
     * @return
     * @throws Exception
     */
    @PostMapping("/readProjectBidEvalCommitteeData")
    public ResponseEntity readProjectBidEvalCommitteeData(String tenderNo) throws Exception {
        String apiMethod = "ReadProjectBidEvalCommitteeData";

        return getResponseEntityByTenderNo(tenderNo, apiMethod);
    }

    /**
     * 2.6 专家抽取组信息表
     *
     * @param tenderNo
     * @return
     * @throws Exception
     */
    @PostMapping("/readProjectCommitteeSelectGroupData")
    public ResponseEntity readProjectCommitteeSelectGroupData(String tenderNo) throws Exception {
        String apiMethod = "ReadProjectCommitteeSelectGroupData";

        return getResponseEntityByTenderNo(tenderNo, apiMethod);
    }

    /**
     * 2.7 专家抽取信息表
     *
     * @param tenderNo
     * @return
     * @throws Exception
     */
    @PostMapping("/readProjectSelectJudgeData")
    public ResponseEntity readProjectSelectJudgeData(String tenderNo) throws Exception {
        String apiMethod = "ReadProjectSelectJudgeData";

        return getResponseEntityByTenderNo(tenderNo, apiMethod);
    }

    /**
     * 2.8 甲方评委信息表
     *
     * @param tenderNo
     * @return
     * @throws Exception
     */
    @PostMapping("/readProjectOwnerJudgeData")
    public ResponseEntity readProjectOwnerJudgeData(String tenderNo) throws Exception {
        String apiMethod = "ReadProjectOwnerJudgeData";

        return getResponseEntityByTenderNo(tenderNo, apiMethod);
    }

    /**
     * 2.9 中标信息表
     *
     * @param tenderNo
     * @return
     * @throws Exception
     */
    @PostMapping("/readTenderWinData")
    public ResponseEntity readTenderWinData(String tenderNo) throws Exception {
        String apiMethod = "ReadTenderWinData";

        return getResponseEntityByTenderNo(tenderNo, apiMethod);
    }


    /**
     * 2.2~2.9接口参数都为 tenderNo 项目编号，这里提取复用
     *
     * @param tenderNo
     * @param apiMethod
     * @return
     * @throws Exception
     */
    private ResponseEntity getResponseEntityByTenderNo(String tenderNo, String apiMethod) throws Exception {

        // 1. 设置参数
        StringBuilder soapRequestData = new StringBuilder();
        appendStartPart(soapRequestData);
        soapRequestData.append("<").append(apiMethod).append(" xmlns=\"http://tempuri.org/\">");
        soapRequestData.append("<userToken>{0}</userToken>");
        soapRequestData.append("<TenderNo>{1}</TenderNo>");
        soapRequestData.append("</").append(apiMethod).append(">");
        appendEndPart(soapRequestData);
        log.info("接收参数 - tenderNo：{}", tenderNo);

        // 2. 参数加密，替换占位符
        String paramData = MessageFormat.format(soapRequestData.toString(),
                TOKEN, AESUtils.desEncrypt(tenderNo));

        // 3. 发送请求
        String resultStr = restTemplateService.sgbSendRequest(paramData, apiMethod);

        // 4. 解析响应并返回
        return getParseResponseXmlAndConvertToJson(apiMethod, resultStr);
    }


    /**
     * 解析接口响应xml，处理异常信息，结果转为json格式返回
     *
     * @param apiMethod
     * @param resultStr
     * @return
     * @throws DocumentException
     */
    private ResponseEntity getParseResponseXmlAndConvertToJson(String apiMethod, String resultStr) throws DocumentException {
        // 标签节点
        String elementResponse = apiMethod + "Response";
        String elementResult = apiMethod + "Result";

        // 1. 解析xml，获取标签数据
        SAXReader read = new SAXReader();
        Document doc = read.read(new ByteArrayInputStream(resultStr.getBytes(StandardCharsets.UTF_8)));
        Element resultElement = doc.getRootElement()
                .element("Body")
                .element(elementResponse)
                .element(elementResult);
        String encryptDataResult = resultElement.getText();

        // 2. 节点数据非加密后的字符串数据时，将接口异常信息返回
        if (StringUtils.isBlank(encryptDataResult)) {
            String dataResult = resultElement.getStringValue();
            return exceptionResponseEntity(dataResult);
        }

        // 3. 解密标签数据（xml形式）
        String decryptDataResult = AESUtils.desDecrypt(encryptDataResult);
        if (StringUtils.isBlank(decryptDataResult)) {
            return exceptionResponseEntity("未获取到数据");
        }

        // 4. 解密后xml结果转为json并返回
        JSONObject jsonResult = XML.toJSONObject(decryptDataResult);
        log.info("节点数据解密转json后：{}", jsonResult);

        // 节点数据非xml，将解密后信息返回
        if (jsonResult.isEmpty()) {
            log.info("节点数据解密后：{}", decryptDataResult);
            return exceptionResponseEntity(decryptDataResult);
        }

        // 返回json信息
        jsonResult.put("success", true);
        return ResponseEntity
                .status(HttpStatus.OK)
                .header(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_VALUE)
                .body(jsonResult.toString());
    }

    /**
     * 异常响应体
     */
    private ResponseEntity exceptionResponseEntity(String errMsg) {
        Map<String, Object> resultBody = new HashMap<>(2);
        resultBody.put("success", false);
        resultBody.put("error", errMsg);
        return ResponseEntity
                .status(HttpStatus.INTERNAL_SERVER_ERROR)
                .header(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_VALUE)
                .body(resultBody);
    }

    /**
     * 拼接xml参数开始部分
     */
    private void appendStartPart(StringBuilder soapRequestData) {
        soapRequestData.append("<?xml version=\"1.0\" encoding=\"utf-8\"?>");
        soapRequestData.append("<soap12:Envelope xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:soap12=\"http://www.w3.org/2003/05/soap-envelope\">");
        soapRequestData.append("<soap12:Body>");
    }

    /**
     * 拼接xml参数结束部分
     */
    private void appendEndPart(StringBuilder soapRequestData) {
        soapRequestData.append("</soap12:Body>");
        soapRequestData.append("</soap12:Envelope>");
    }
}
