package com.mth.requestsecret.service;

import com.alibaba.fastjson.JSONObject;
import com.mth.requestsecret.scheduler.RequestSecretSchedulerTask;
import com.mth.requestsecret.util.AESUtils;
import com.mth.requestsecret.util.DSLUtils;
import com.mth.requestsecret.util.MD5Utils;
import com.mth.requestsecret.util.RedisUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringEscapeUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.*;
import org.springframework.stereotype.Component;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestTemplate;

import java.util.Date;
import java.util.Objects;

import static com.mth.requestsecret.constant.Constants.RESPONSE_CODE_14;
import static com.mth.requestsecret.constant.Constants.SJJ_REQUEST_SECRET_PREFIX;

/**
 * @author fyl
 * @version v1.0
 * @desc
 * @date 2020-07-16 15:00
 * @jdk 1.8
 */
@Component
@Slf4j
public class RestTemplateService {

    @Autowired
    private RestTemplate restTemplate;

    @Autowired
    private RedisUtils redisUtil;

    @Autowired
    private RequestSecretSchedulerTask schedulerTask;

    private ThreadLocal<Integer> threadLocal = ThreadLocal.withInitial(() -> 0);

    @Value("${ruicheng.address}")
    private String pathUrl;

    @Value("${ruicheng.key}")
    private String appKey;

    @Value("${zbd.appid}")
    private String zbdAppid;

    @Value("${zbd.appSecret}")
    private String zbdAppSecret;

    @Value("${zbd.address}")
    private String zbdAddress;

    @Value("${sgb.address}")
    private String sgbAddress;

    /**
     * 瑞成平台接口发送请求
     *
     * @param paramMap
     * @param apiMethod
     * @return
     */
    public ResponseEntity<String> commonSendRequest(MultiValueMap<String, Object> paramMap, final String apiMethod) {
        // 请求url拼接api签名和公共参数
        StringBuilder url = new StringBuilder()
                .append(pathUrl)
                .append("/interface/public/service/risen-api/")
                .append(apiMethod);

        // 请求时间
        String requestTime = DSLUtils.dateToLong(new Date()) + "";
        // redis中获取秘钥
        String requestSecret = redisUtil.get(SJJ_REQUEST_SECRET_PREFIX);
        log.info("redis中请求秘钥：{}", requestSecret);
        // 签名字符串
        String signStr = appKey + requestSecret + requestTime;
        // 组装请求参数
        paramMap.set("appKey", appKey);
        paramMap.set("sign", MD5Utils.encoderByMd5(signStr));
        paramMap.set("requestTime", requestTime);

        // 日志记录
        log.info("纪委api：{}", apiMethod);
        log.info("api url：{}", url);
        log.info("api params：{}", paramMap);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_FORM_URLENCODED);

        // 发送请求
        HttpEntity<MultiValueMap<String, Object>> request = new HttpEntity<>(paramMap, headers);
        ResponseEntity<String> responseEntity = restTemplate.exchange(url.toString(), HttpMethod.POST, request, String.class);
        log.info("api response：{}", responseEntity);

        try {
            // 接口返回码
            String code = JSONObject.parseObject(responseEntity.getBody()).getString("code");
            Integer count = threadLocal.get();
            String threadName = Thread.currentThread().getName();

            // 如返回签名错误，则重新获取秘钥并发送请求，限制重新请求次数两次
            if (RESPONSE_CODE_14.equals(code) && count < 2) {
                threadLocal.set(++count);
                log.info("线程名称：{}，重新请求次数：{}", threadName, count);
                // 获取秘钥
                schedulerTask.getRequestSecret();
                // 重新调用请求
                return commonSendRequest(paramMap, apiMethod);
            }
        } finally {
            threadLocal.remove();
        }
        return responseEntity;
    }


    /**
     * 招必得平台接口发送请求
     */
    public ResponseEntity<String> zbdSendRequest(JSONObject paramMap, final String apiMethod) {

        // 1. 获取请求时间戳并解密
        String timeStamp = getZbdTimeStampAndDecrypt();

        // 2. 拼接header验证签名
        String signStr = zbdAppid + zbdAppSecret + timeStamp;
        String sign = MD5Utils.encoderByMd5UpperCase(signStr);
        // authorization : appid=xxx&sign=xxx
        String authorization = "appid=" + zbdAppid + "&sign=" + sign;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.set("authorization", authorization);

        // 3. 发送请求
        HttpEntity<String> request = new HttpEntity<>(paramMap.toJSONString(), headers);
        String url = zbdAddress + apiMethod;
        ResponseEntity<String> responseEntity = restTemplate.exchange(url, HttpMethod.POST, request, String.class);

        // 4. 日志记录
        log.info("招必得api：{}", apiMethod);
        log.info("api url：{}", url);
        log.info("api params：{}", paramMap);
        log.info("api response：{}", responseEntity);

        return responseEntity;
    }

    /**
     * 获取请求时间戳并解密
     *
     * @return
     */
    private String getZbdTimeStampAndDecrypt() {
        String timeStampUrl = zbdAddress + "/gen/querytimestamp.do?appid=2020081001";
        ResponseEntity<String> timeStampResponse = restTemplate.getForEntity(timeStampUrl, String.class);
        JSONObject body = JSONObject.parseObject(timeStampResponse.getBody());
        if (!(Objects.nonNull(body) && Objects.equals("true", body.getString("success")))) {
            log.warn("招必得请求时间戳接口返回异常：{}", timeStampResponse);
            throw new RuntimeException("时间戳接口异常");
        }
        // 解析返回值
        String encryptTimestamp = body.getJSONObject("data").getString("encrypt_key");
        // 解密
        return AESUtils.decrypt(encryptTimestamp);
    }

    /**
     * 审管办接口发送请求（webService）
     *
     * @param soapRequestData
     * @param apiMethod
     * @return
     */
    public String sgbSendRequest(String soapRequestData, String apiMethod) throws Exception {

        // 1. 构造http请求头
        HttpHeaders headers = new HttpHeaders();
        MediaType type = MediaType.parseMediaType("text/xml;charset=UTF-8");
        headers.setContentType(type);
        HttpEntity<String> formEntity = new HttpEntity<String>(soapRequestData, headers);

        // 2. 返回结果
        String resultStr = restTemplate.postForObject(sgbAddress, formEntity, String.class);

        // 3. 日志记录
        log.info("审管办api：{}", apiMethod);
        log.info("api url：{}", sgbAddress);
        log.info("api params：{}", soapRequestData);
        log.info("api response：{}", resultStr);

        // 转换返回结果中的特殊字符，返回的结果中会将xml转义，此处需要反转移
        return StringEscapeUtils.unescapeXml(resultStr);
    }

}